// Import required modules
const express = require('express');
const http = require('http');
const socketIO = require('socket.io');
const path = require('path');

// Initialize Express app and HTTP server
const app = express();
const server = http.createServer(app);

// Initialize Socket.IO with CORS configuration for local development
const io = socketIO(server, {
    cors: {
        origin: "*",
        methods: ["GET", "POST"]
    }
});

// Serve static files from the public directory
app.use(express.static('public'));

// Serve the join page (index.html)
app.get('/', (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

// Serve the chat page
app.get('/chat.html', (req, res) => {
    res.sendFile(path.join(__dirname, 'public', 'chat.html'));
});

// API endpoint to validate username availability
app.get('/api/validate-username', (req, res) => {
    const username = req.query.username;

    // Validate username format
    if (!username || username.trim().length === 0) {
        return res.json({
            available: false,
            message: 'Username cannot be empty'
        });
    }

    const trimmedUsername = username.trim();

    if (trimmedUsername.length > 20) {
        return res.json({
            available: false,
            message: 'Username must be 20 characters or less'
        });
    }

    // Check if username is already taken
    const usernameTaken = Array.from(connectedUsers.values()).some(
        userData => userData.username.toLowerCase() === trimmedUsername.toLowerCase()
    );

    if (usernameTaken) {
        return res.json({
            available: false,
            message: `Username "${trimmedUsername}" is already taken. Please choose another name.`
        });
    }

    // Username is available
    res.json({
        available: true,
        message: 'Username is available'
    });
});


// In-memory storage for connected users (socketId -> user data object)
const connectedUsers = new Map();

// In-memory storage for message reactions (messageId -> {emoji -> Set(userIds)})
const messageReactions = new Map();

// In-memory storage for emoji leaderboard (userId -> count)
const emojiLeaderboard = new Map();

// Helper function to generate random color
function generateUserColor() {
    const colors = [
        '#e74c3c', '#3498db', '#2ecc71', '#f39c12', '#9b59b6',
        '#1abc9c', '#e67e22', '#34495e', '#16a085', '#27ae60',
        '#2980b9', '#8e44ad', '#c0392b', '#d35400', '#7f8c8d'
    ];
    return colors[Math.floor(Math.random() * colors.length)];
}

// WebSocket connection handling
io.on('connection', (socket) => {
    console.log('New client connected:', socket.id);

    // Handle user joining the chat
    socket.on('user-join', (username) => {
        // Validate username
        if (!username || username.trim().length === 0) {
            socket.emit('join-error', { message: 'Username cannot be empty' });
            return;
        }

        const trimmedUsername = username.trim();

        // Check if this socket has already joined
        if (connectedUsers.has(socket.id)) {
            const currentUserData = connectedUsers.get(socket.id);
            socket.emit('join-error', {
                message: `You are already in the chat as "${currentUserData.username}". Please refresh the page to change your username.`
            });
            console.log(`Socket ${socket.id} attempted to rejoin (already in as ${currentUserData.username})`);
            return;
        }

        // Check if username is already taken by another user
        const usernameTaken = Array.from(connectedUsers.values()).some(
            userData => userData.username.toLowerCase() === trimmedUsername.toLowerCase()
        );

        if (usernameTaken) {
            socket.emit('join-error', {
                message: `Username "${trimmedUsername}" is already taken. Please choose another name.`
            });
            console.log(`Username "${trimmedUsername}" is already in use`);
            return;
        }

        // Generate random color for user
        const userColor = generateUserColor();

        // Initialize emoji count for this user
        emojiLeaderboard.set(socket.id, 0);

        // Store the user data associated with this socket
        const userData = {
            username: trimmedUsername,
            color: userColor,
            emojiCount: 0
        };

        connectedUsers.set(socket.id, userData);

        console.log(`${trimmedUsername} joined the chat with color ${userColor}`);

        // Send success confirmation to the user who joined
        socket.emit('join-success', {
            username: trimmedUsername,
            color: userColor
        });

        // Broadcast to all clients that a new user has joined
        io.emit('user-joined', {
            username: trimmedUsername,
            userId: socket.id,
            color: userColor,
            timestamp: new Date().toISOString()
        });

        // Send current user count to all clients
        io.emit('user-count', connectedUsers.size);

        // Broadcast online users list to all clients
        const onlineUsers = Array.from(connectedUsers.entries()).map(([socketId, userData]) => ({
            userId: socketId,
            username: userData.username,
            color: userData.color,
            emojiCount: userData.emojiCount
        }));
        io.emit('online-users', onlineUsers);

        // Broadcast emoji leaderboard
        broadcastEmojiLeaderboard();
    });

    // Handle incoming messages (both emoji and text)
    socket.on('send-message', (data) => {
        const userData = connectedUsers.get(socket.id);

        if (userData) {
            // Validate message data
            if (!data || !data.type || !data.content) {
                console.log('Invalid message data received');
                return;
            }

            // Validate message type
            if (data.type !== 'emoji' && data.type !== 'text') {
                console.log('Invalid message type:', data.type);
                return;
            }

            // Validate content length for text messages
            if (data.type === 'text' && data.content.length > 500) {
                socket.emit('message-error', { message: 'Message too long (max 500 characters)' });
                return;
            }

            // Track emoji count if it's an emoji message
            if (data.type === 'emoji') {
                userData.emojiCount++;
                emojiLeaderboard.set(socket.id, userData.emojiCount);

                // Broadcast updated leaderboard
                broadcastEmojiLeaderboard();

                // Update online users list with new emoji count
                const onlineUsers = Array.from(connectedUsers.entries()).map(([socketId, userData]) => ({
                    userId: socketId,
                    username: userData.username,
                    color: userData.color,
                    emojiCount: userData.emojiCount
                }));
                io.emit('online-users', onlineUsers);
            }

            // Create message object with all necessary data
            const message = {
                username: userData.username,
                type: data.type,
                content: data.content,
                timestamp: new Date().toISOString(),
                userId: socket.id,
                color: userData.color
            };

            console.log(`${userData.username} sent ${data.type}: ${data.content.substring(0, 50)}`);

            // Broadcast the message to all connected clients
            io.emit('new-message', message);
        }
    });

    // Handle emoji reactions on messages
    socket.on('add-reaction', (data) => {
        const userData = connectedUsers.get(socket.id);

        if (userData && data && data.messageId && data.emoji) {
            // Initialize reactions for this message if not exists
            if (!messageReactions.has(data.messageId)) {
                messageReactions.set(data.messageId, new Map());
            }

            const msgReactions = messageReactions.get(data.messageId);

            // Initialize emoji set if not exists
            if (!msgReactions.has(data.emoji)) {
                msgReactions.set(data.emoji, new Set());
            }

            const userIds = msgReactions.get(data.emoji);

            // Toggle reaction: if user already reacted, remove it; otherwise add it
            if (userIds.has(socket.id)) {
                userIds.delete(socket.id);
                // If no users left for this emoji, remove the emoji key
                if (userIds.size === 0) {
                    msgReactions.delete(data.emoji);
                }
            } else {
                // Check if user has reacted with other emojis on this message? 
                // Requirement says "react only once per emoji per message", which implies multiple different emojis are allowed?
                // "Users can react only once per emoji per message" - this usually means I can't click '👍' twice to give 2 thumbs up.
                // It doesn't explicitly say "one reaction per message per user".
                // So I will allow multiple different emojis, but toggle individual ones.
                userIds.add(socket.id);
            }

            // Broadcast the updated reactions to all clients
            const reactionsObject = {};
            msgReactions.forEach((userIds, emoji) => {
                reactionsObject[emoji] = userIds.size;
            });

            io.emit('reaction-update', {
                messageId: data.messageId,
                reactions: reactionsObject
            });

            console.log(`${userData.username} toggled reaction ${data.emoji} on message ${data.messageId}`);
        }
    });

    // Handle typing indicators
    socket.on('typing-start', () => {
        const userData = connectedUsers.get(socket.id);
        if (userData) {
            socket.broadcast.emit('user-typing', { username: userData.username, userId: socket.id });
        }
    });

    socket.on('typing-stop', () => {
        const userData = connectedUsers.get(socket.id);
        if (userData) {
            socket.broadcast.emit('user-stopped-typing', { userId: socket.id });
        }
    });

    // Handle disconnection
    socket.on('disconnect', () => {
        const userData = connectedUsers.get(socket.id);

        if (userData) {
            console.log(`${userData.username} disconnected`);

            // Remove user from connected users and leaderboard
            connectedUsers.delete(socket.id);
            emojiLeaderboard.delete(socket.id);

            // Notify all clients that user has left
            io.emit('user-left', {
                username: userData.username,
                userId: socket.id,
                timestamp: new Date().toISOString()
            });

            // Update user count
            io.emit('user-count', connectedUsers.size);

            // Broadcast updated online users list
            const onlineUsers = Array.from(connectedUsers.entries()).map(([socketId, userData]) => ({
                userId: socketId,
                username: userData.username,
                color: userData.color,
                emojiCount: userData.emojiCount
            }));
            io.emit('online-users', onlineUsers);

            // Broadcast updated leaderboard
            broadcastEmojiLeaderboard();
        }
    });
});

// Helper function to broadcast emoji leaderboard
function broadcastEmojiLeaderboard() {
    const leaderboard = Array.from(connectedUsers.entries())
        .map(([socketId, userData]) => ({
            username: userData.username,
            emojiCount: userData.emojiCount,
            color: userData.color
        }))
        .sort((a, b) => b.emojiCount - a.emojiCount)
        .slice(0, 5); // Top 5 users

    io.emit('emoji-leaderboard', leaderboard);
}

// Start the server
const PORT = process.env.PORT || 3000;
server.listen(PORT, () => {
    console.log(`🚀 Emoji & Text Chat Server running on http://localhost:${PORT}`);
    console.log(`📱 Open multiple browser tabs to test real-time chat!`);
});
