// Join page JavaScript

// DOM elements
const usernameInput = document.getElementById('username-input');
const joinBtn = document.getElementById('join-btn');
const errorMessage = document.getElementById('error-message');

// Redirect to chat if already has username in session
window.addEventListener('DOMContentLoaded', () => {
    const savedUsername = sessionStorage.getItem('chatUsername');
    if (savedUsername) {
        // User already has a session, redirect to chat
        window.location.href = '/chat.html';
    }
});

// Join chat function with server-side validation
async function joinChat() {
    const username = usernameInput.value.trim();

    // Clear any previous error messages
    errorMessage.textContent = '';
    errorMessage.style.display = 'none';

    // Basic client-side validation
    if (username.length < 1) {
        showError('Please enter a username');
        return;
    }

    if (username.length > 20) {
        showError('Username must be 20 characters or less');
        return;
    }

    // Disable the button while checking
    joinBtn.disabled = true;
    joinBtn.textContent = 'Checking...';

    try {
        // Validate username with server
        const response = await fetch(`/api/validate-username?username=${encodeURIComponent(username)}`);
        const data = await response.json();

        if (data.available) {
            // Username is available, store and redirect
            sessionStorage.setItem('chatUsername', username);
            window.location.href = '/chat.html';
        } else {
            // Username is taken or invalid
            showError(data.message);
            joinBtn.disabled = false;
            joinBtn.textContent = 'Join Chat';
        }
    } catch (error) {
        console.error('Error validating username:', error);
        showError('Error connecting to server. Please try again.');
        joinBtn.disabled = false;
        joinBtn.textContent = 'Join Chat';
    }
}

// Show error message
function showError(message) {
    errorMessage.textContent = message;
    errorMessage.style.display = 'block';
    usernameInput.focus();
}

// Event listeners
joinBtn.addEventListener('click', joinChat);

// Allow Enter key to join
usernameInput.addEventListener('keypress', (e) => {
    if (e.key === 'Enter') {
        joinChat();
    }
});

// Clear error message when user starts typing
usernameInput.addEventListener('input', () => {
    if (errorMessage.style.display === 'block') {
        errorMessage.style.display = 'none';
    }
});
