// Chat page JavaScript

// Emoji collection organized by categories
const emojiCategories = {
    'Smileys & People': [
        '😀', '😃', '😄', '😁', '😆', '😅', '🤣', '😂',
        '🙂', '🙃', '😉', '😊', '😇', '🥰', '😍', '🤩',
        '😘', '😗', '😚', '😙', '😋', '😛', '😜', '🤪',
        '😝', '🤑', '🤗', '🤭', '🤫', '🤔', '🤐', '🤨',
        '😐', '😑', '😶', '😏', '😒', '🙄', '😬', '🤥',
        '😌', '😔', '😪', '🤤', '😴', '😷', '🤒', '🤕'
    ],
    'Animals & Nature': [
        '🐶', '🐱', '🐭', '🐹', '🐰', '🦊', '🐻', '🐼',
        '🐨', '🐯', '🦁', '🐮', '🐷', '🐸', '🐵', '🐔',
        '🐧', '🐦', '🐤', '🦆', '🦅', '🦉', '🦇', '🐺',
        '🐗', '🐴', '🦄', '🐝', '🐛', '🦋', '🐌', '🐞',
        '🌸', '🌺', '🌻', '🌷', '🌹', '🌼', '🌱', '🌲',
        '🌳', '🌴', '🌵', '🌾', '🌿', '☘️', '🍀', '🍁'
    ],
    'Food & Drink': [
        '🍎', '🍊', '🍋', '🍌', '🍉', '🍇', '🍓', '🍈',
        '🍒', '🍑', '🥭', '🍍', '🥥', '🥝', '🍅', '🍆',
        '🥑', '🥦', '🥬', '🥒', '🌶️', '🌽', '🥕', '🥔',
        '🍞', '🥐', '🥖', '🥨', '🥯', '🧀', '🥚', '🍳',
        '🥞', '🧇', '🥓', '🍔', '🍟', '🍕', '🌭', '🥪',
        '🌮', '🌯', '🥙', '🧆', '🍿', '🧈', '☕', '🍵'
    ],
    'Activities & Sports': [
        '⚽', '🏀', '🏈', '⚾', '🥎', '🎾', '🏐', '🏉',
        '🥏', '🎱', '🏓', '🏸', '🏒', '🏑', '🥍', '🏏',
        '🥅', '⛳', '🏹', '🎣', '🤿', '🥊', '🥋', '🎽',
        '🛹', '🛷', '⛸️', '🥌', '🎿', '⛷️', '🏂', '🪂',
        '🏋️', '🤸', '🤺', '🤾', '🏌️', '🏇', '🧘', '🏄',
        '🎮', '🎯', '🎲', '🎰', '🎳', '🎨', '🎭', '🎪'
    ],
    'Travel & Places': [
        '🚗', '🚕', '🚙', '🚌', '🚎', '🏎️', '🚓', '🚑',
        '🚒', '🚐', '🚚', '🚛', '🚜', '🏍️', '🛵', '🚲',
        '🛴', '🚁', '🛩️', '✈️', '🚀', '🛸', '🚂', '🚆',
        '🚇', '🚊', '🚝', '🚄', '🚅', '🚈', '🚞', '🚋',
        '🏠', '🏡', '🏢', '🏣', '🏤', '🏥', '🏦', '🏨',
        '🏩', '🏪', '🏫', '🏬', '🏭', '🏯', '🏰', '🗼'
    ],
    'Objects': [
        '⌚', '📱', '💻', '⌨️', '🖥️', '🖨️', '🖱️', '🖲️',
        '🕹️', '🗜️', '💾', '💿', '📀', '📼', '📷', '📸',
        '📹', '🎥', '📞', '☎️', '📟', '📠', '📺', '📻',
        '🎙️', '🎚️', '🎛️', '🧭', '⏱️', '⏲️', '⏰', '🕰️',
        '💡', '🔦', '🕯️', '🪔', '🧯', '🛢️', '💸', '💵',
        '💴', '💶', '💷', '💰', '💳', '🎁', '🎀', '🎊'
    ],
    'Symbols': [
        '❤️', '🧡', '💛', '💚', '💙', '💜', '🖤', '🤍',
        '🤎', '💔', '❣️', '💕', '💞', '💓', '💗', '💖',
        '💘', '💝', '💟', '☮️', '✝️', '☪️', '🕉️', '☸️',
        '✡️', '🔯', '🕎', '☯️', '☦️', '🛐', '⛎', '♈',
        '♉', '♊', '♋', '♌', '♍', '♎', '♏', '♐',
        '♑', '♒', '♓', '🆔', '⚛️', '🉑', '☢️', '☣️',
        '⭐', '🌟', '✨', '⚡', '🔥', '💥', '💯', '✅'
    ]
};

// Flatten all emojis for the quick send palette (first 30)
const quickEmojis = Object.values(emojiCategories).flat().slice(0, 30);

// Reaction emojis (for quick reactions)
const reactionEmojis = ['👍', '❤️', '😂', '😮', '😢', '🔥'];

// Socket.IO connection
let socket;
let currentUsername = '';
let currentUserId = '';
let currentUserColor = '';
let isConnected = false;
let typingTimeout;

// DOM elements
const currentUsernameDisplay = document.getElementById('current-username');
const userCountDisplay = document.getElementById('user-count');
const chatMessages = document.getElementById('chat-messages');
const emojiGrid = document.getElementById('emoji-grid');
const emojiCategoryTabs = document.getElementById('emoji-category-tabs');
const textInput = document.getElementById('text-input');
const sendTextBtn = document.getElementById('send-text-btn');
const leaveBtn = document.getElementById('leave-btn');
const typingIndicator = document.getElementById('typing-indicator');
const onlineUsersList = document.getElementById('online-users-list');
const emojiLeaderboardEl = document.getElementById('emoji-leaderboard');

// Initialize the application
function init() {
    // Check if user has username in session
    const savedUsername = sessionStorage.getItem('chatUsername');

    if (!savedUsername) {
        // No username in session, redirect to join page
        window.location.href = '/';
        return;
    }

    currentUsername = savedUsername;
    currentUsernameDisplay.textContent = `Logged in as: ${currentUsername}`;

    // Create emoji category tabs
    createEmojiCategoryTabs();

    // Populate emoji palette with first category
    const firstCategory = Object.keys(emojiCategories)[0];
    populateEmojiGrid(firstCategory);

    // Initialize Socket.IO connection
    socket = io();

    // Set up socket event listeners
    setupSocketListeners();

    // Emit user join event to server
    socket.emit('user-join', currentUsername);

    // Set up text input event listeners
    sendTextBtn.addEventListener('click', sendTextMessage);
    textInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            sendTextMessage();
        }
    });

    // Typing indicator
    textInput.addEventListener('input', () => {
        if (!typingTimeout) {
            socket.emit('typing-start');
        }
        clearTimeout(typingTimeout);
        typingTimeout = setTimeout(() => {
            socket.emit('typing-stop');
            typingTimeout = null;
        }, 1000);
    });

    // Leave button
    leaveBtn.addEventListener('click', leaveChat);
}

// Create emoji category tabs
function createEmojiCategoryTabs() {
    emojiCategoryTabs.innerHTML = '';
    Object.keys(emojiCategories).forEach(category => {
        const tab = document.createElement('button');
        tab.className = 'emoji-category-tab';
        tab.textContent = category;
        tab.onclick = () => {
            // Remove active class from all tabs
            document.querySelectorAll('.emoji-category-tab').forEach(t => t.classList.remove('active'));
            tab.classList.add('active');
            populateEmojiGrid(category);
        };
        emojiCategoryTabs.appendChild(tab);
    });
    // Set first tab as active
    emojiCategoryTabs.firstChild.classList.add('active');
}

// Populate emoji grid with category emojis
function populateEmojiGrid(category) {
    emojiGrid.innerHTML = '';
    emojiCategories[category].forEach(emoji => {
        const button = document.createElement('button');
        button.className = 'emoji-btn';
        button.textContent = emoji;
        button.onclick = () => sendEmoji(emoji);
        emojiGrid.appendChild(button);
    });
}

// Set up all Socket.IO event listeners
function setupSocketListeners() {
    // Listen for successful join confirmation
    socket.on('join-success', (data) => {
        isConnected = true;
        currentUserId = socket.id;
        currentUserColor = data.color;
        currentUserColor = data.color;
        currentUsernameDisplay.textContent = `${data.username}`;
        currentUsernameDisplay.style.color = data.color;
        displaySystemMessage('You joined the chat');
    });

    // Listen for join errors (duplicate username or already joined)
    socket.on('join-error', (data) => {
        alert(data.message);
        // Clear session and redirect to join page
        sessionStorage.removeItem('chatUsername');
        window.location.href = '/';
    });

    // Listen for new messages from any user (both emoji and text)
    socket.on('new-message', (message) => {
        displayMessage(message);
    });

    // Listen for user joined notifications
    socket.on('user-joined', (data) => {
        displaySystemMessage(`${data.username} joined the chat`);
    });

    // Listen for user left notifications
    socket.on('user-left', (data) => {
        displaySystemMessage(`${data.username} left the chat`);
    });

    // Listen for user count updates
    socket.on('user-count', (count) => {
        userCountDisplay.textContent = count;
    });

    // Listen for online users list updates
    socket.on('online-users', (users) => {
        updateOnlineUsersList(users);
    });

    // Listen for emoji leaderboard updates
    socket.on('emoji-leaderboard', (leaderboard) => {
        updateEmojiLeaderboard(leaderboard);
    });

    // Listen for reaction updates
    socket.on('reaction-update', (data) => {
        updateMessageReactions(data.messageId, data.reactions);
    });

    // Listen for typing indicators
    socket.on('user-typing', (data) => {
        showTypingIndicator(data.username);
    });

    socket.on('user-stopped-typing', (data) => {
        hideTypingIndicator(data.userId);
    });

    // Handle connection errors
    socket.on('connect_error', (error) => {
        console.error('Connection error:', error);
        displaySystemMessage('Connection error. Please refresh the page.');
        isConnected = false;
    });

    // Handle disconnection
    socket.on('disconnect', () => {
        displaySystemMessage('Disconnected from server');
        isConnected = false;
    });
}

// Send a text message
function sendTextMessage() {
    const text = textInput.value.trim();

    if (text.length === 0) {
        return; // Don't send empty messages
    }

    if (text.length > 500) {
        alert('Message is too long (max 500 characters)');
        return;
    }

    if (socket && socket.connected) {
        // Emit the text message to the server
        socket.emit('send-message', {
            type: 'text',
            content: text
        });

        // Clear input field
        textInput.value = '';

        // Stop typing indicator
        socket.emit('typing-stop');
        clearTimeout(typingTimeout);
        typingTimeout = null;
    } else {
        displaySystemMessage('Not connected to server');
    }
}

// Send an emoji message
function sendEmoji(emoji) {
    if (socket && socket.connected) {
        // Emit the emoji message to the server
        socket.emit('send-message', {
            type: 'emoji',
            content: emoji
        });
    } else {
        displaySystemMessage('Not connected to server');
    }
}

// Display a message in the chat (emoji or text)
function displayMessage(message) {
    const messageDiv = document.createElement('div');
    messageDiv.className = 'message';
    messageDiv.dataset.messageId = message.timestamp + '_' + message.userId;

    // Add class if this is the current user's message
    if (message.userId === currentUserId) {
        messageDiv.classList.add('own-message');
    }

    // Format timestamp
    const time = new Date(message.timestamp).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit'
    });

    // Different display based on message type
    let contentHtml = '';
    if (message.type === 'emoji') {
        contentHtml = `<div class="message-emoji">${message.content}</div>`;
    } else if (message.type === 'text') {
        contentHtml = `<div class="message-text">${escapeHtml(message.content)}</div>`;
    }

    messageDiv.innerHTML = `
        <div class="message-header">
            <span class="message-username" style="color: ${message.color}">${escapeHtml(message.username)}</span>
            <span class="message-time">${time}</span>
        </div>
        ${contentHtml}
        <div class="message-reactions" id="reactions-${message.timestamp}_${message.userId}">
        </div>
        <div class="reaction-picker">
            ${reactionEmojis.map(emoji => `
                <button class="add-reaction-btn" onclick="addReaction('${message.timestamp}_${message.userId}', '${emoji}')" title="React with ${emoji}">${emoji}</button>
            `).join('')}
        </div>
    `;

    chatMessages.appendChild(messageDiv);

    // Auto-scroll to the latest message
    scrollToBottom();
}

// Add a reaction to a message
function addReaction(messageId, emoji) {
    if (socket && socket.connected) {
        socket.emit('add-reaction', {
            messageId: messageId,
            emoji: emoji
        });
    }
}

// Update message reactions display
function updateMessageReactions(messageId, reactions) {
    const reactionsContainer = document.getElementById(`reactions-${messageId}`);
    if (!reactionsContainer) return;

    reactionsContainer.innerHTML = '';

    Object.entries(reactions).forEach(([emoji, count]) => {
        const reactionBtn = document.createElement('div');
        reactionBtn.className = 'reaction-btn';
        reactionBtn.innerHTML = `
            <span>${emoji}</span>
            <span class="reaction-count">${count}</span>
        `;
        reactionsContainer.appendChild(reactionBtn);
    });
}

// Update online users list
function updateOnlineUsersList(users) {
    onlineUsersList.innerHTML = '';

    users.forEach(user => {
        const userItem = document.createElement('div');
        userItem.className = 'online-user-item';

        if (user.userId === currentUserId) {
            userItem.classList.add('current-user-item');
        }

        userItem.innerHTML = `
            <span class="user-status-dot"></span>
            <span style="color: ${user.color}; font-weight: bold;">${escapeHtml(user.username)}</span>
            ${user.userId === currentUserId ? '<span style="margin-left: auto; font-size: 0.8em;">(You)</span>' : ''}
            <span class="user-emoji-count" title="Emojis sent">🎯 ${user.emojiCount}</span>
        `;

        onlineUsersList.appendChild(userItem);
    });
}

// Update emoji leaderboard
function updateEmojiLeaderboard(leaderboard) {
    emojiLeaderboardEl.innerHTML = '<h4>🏆 Top Emoji Senders</h4>';

    if (leaderboard.length === 0) {
        emojiLeaderboardEl.innerHTML += '<p class="no-data">No emojis sent yet!</p>';
        return;
    }

    const leaderboardList = document.createElement('div');
    leaderboardList.className = 'leaderboard-list';

    leaderboard.forEach((user, index) => {
        const medal = index === 0 ? '🥇' : index === 1 ? '🥈' : index === 2 ? '🥉' : '🏅';
        const item = document.createElement('div');
        item.className = 'leaderboard-item';
        item.innerHTML = `
            <span class="leaderboard-rank">${medal}</span>
            <span class="leaderboard-name" style="color: ${user.color}">${escapeHtml(user.username)}</span>
            <span class="leaderboard-count">${user.emojiCount}</span>
        `;
        leaderboardList.appendChild(item);
    });

    emojiLeaderboardEl.appendChild(leaderboardList);
}

// Show typing indicator
const typingUsers = new Set();

function showTypingIndicator(username) {
    typingUsers.add(username);
    updateTypingIndicator();
}

function hideTypingIndicator(userId) {
    // We'll just clear all after a timeout since we only have userId
    setTimeout(() => {
        updateTypingIndicator();
    }, 1500);
}

function updateTypingIndicator() {
    if (typingUsers.size === 0) {
        typingIndicator.textContent = '';
        return;
    }

    const names = Array.from(typingUsers).slice(0, 3);
    if (typingUsers.size === 1) {
        typingIndicator.textContent = `${names[0]} is typing...`;
    } else if (typingUsers.size === 2) {
        typingIndicator.textContent = `${names[0]} and ${names[1]} are typing...`;
    } else {
        typingIndicator.textContent = `${names[0]}, ${names[1]} and others are typing...`;
    }

    // Clear typing users after 3 seconds
    setTimeout(() => {
        typingUsers.clear();
        typingIndicator.textContent = '';
    }, 3000);
}

// Display a system message (user joined/left)
function displaySystemMessage(text) {
    const messageDiv = document.createElement('div');
    messageDiv.className = 'system-message';
    messageDiv.textContent = text;
    chatMessages.appendChild(messageDiv);

    // Auto-scroll to the latest message
    scrollToBottom();
}

// Scroll chat to bottom (latest message)
function scrollToBottom() {
    chatMessages.scrollTop = chatMessages.scrollHeight;
}

// Escape HTML to prevent XSS attacks
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Leave chat and return to join page
function leaveChat() {
    if (socket) {
        socket.disconnect();
    }
    sessionStorage.removeItem('chatUsername');
    window.location.href = '/';
}

// Initialize the app when DOM is loaded
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
} else {
    init();
}
