// Emoji collection (20+ emojis for the palette)
const emojis = [
    '😀', '😂', '😍', '🥰', '😎', '🤔', '😢', '😭',
    '😡', '🤯', '🥳', '💪', '👍', '👏', '🙏', '❤️',
    '🔥', '✨', '🎉', '🎊', '⭐', '💯', '🚀', '🌈',
    '☕', '🍕', '🎮', '🎵', '📱', '💻'
];

// Socket.IO connection
let socket;
let currentUsername = '';
let isConnected = false; // Track if user has already joined

// DOM elements
const usernameScreen = document.getElementById('username-screen');
const chatScreen = document.getElementById('chat-screen');
const usernameInput = document.getElementById('username-input');
const joinBtn = document.getElementById('join-btn');
const currentUsernameDisplay = document.getElementById('current-username');
const userCountDisplay = document.getElementById('user-count');
const chatMessages = document.getElementById('chat-messages');
const emojiGrid = document.getElementById('emoji-grid');

// Initialize the application
function init() {
    // Populate emoji palette
    emojis.forEach(emoji => {
        const button = document.createElement('button');
        button.className = 'emoji-btn';
        button.textContent = emoji;
        button.onclick = () => sendEmoji(emoji);
        emojiGrid.appendChild(button);
    });

    // Join chat on button click
    joinBtn.addEventListener('click', joinChat);

    // Allow Enter key to join chat
    usernameInput.addEventListener('keypress', (e) => {
        if (e.key === 'Enter') {
            joinChat();
        }
    });
}

// Join the chat room
function joinChat() {
    // Check if already connected
    if (isConnected) {
        alert('You are already in the chat. Please refresh the page to change your username.');
        return;
    }

    const username = usernameInput.value.trim();

    // Validate username
    if (username.length < 1) {
        alert('Please enter a username');
        return;
    }

    if (username.length > 20) {
        alert('Username must be 20 characters or less');
        return;
    }

    // Disable join button to prevent multiple clicks
    joinBtn.disabled = true;
    joinBtn.textContent = 'Joining...';

    currentUsername = username;

    // Initialize Socket.IO connection if not already initialized
    if (!socket) {
        socket = io();
        // Set up socket event listeners
        setupSocketListeners();
    }

    // Emit user join event to server
    socket.emit('user-join', currentUsername);
}

// Set up all Socket.IO event listeners
function setupSocketListeners() {
    // Listen for successful join confirmation
    socket.on('join-success', (data) => {
        isConnected = true;
        // Update UI to show chat screen
        currentUsernameDisplay.textContent = data.username;
        usernameScreen.classList.add('hidden');
        chatScreen.classList.remove('hidden');
        // Re-enable button in case user wants to refresh
        joinBtn.disabled = false;
        joinBtn.textContent = 'Join Chat';
    });

    // Listen for join errors (duplicate username or already joined)
    socket.on('join-error', (data) => {
        alert(data.message);
        // Re-enable join button
        joinBtn.disabled = false;
        joinBtn.textContent = 'Join Chat';
    });

    // Listen for new messages from any user
    socket.on('new-message', (message) => {
        displayMessage(message);
    });

    // Listen for user joined notifications
    socket.on('user-joined', (data) => {
        displaySystemMessage(`${data.username} joined the chat`);
    });

    // Listen for user left notifications
    socket.on('user-left', (data) => {
        displaySystemMessage(`${data.username} left the chat`);
    });

    // Listen for user count updates
    socket.on('user-count', (count) => {
        userCountDisplay.textContent = count;
    });

    // Handle connection errors
    socket.on('connect_error', (error) => {
        console.error('Connection error:', error);
        displaySystemMessage('Connection error. Please refresh the page.');
        isConnected = false;
        joinBtn.disabled = false;
        joinBtn.textContent = 'Join Chat';
    });

    // Handle disconnection
    socket.on('disconnect', () => {
        displaySystemMessage('Disconnected from server');
        isConnected = false;
    });
}

// Send an emoji message
function sendEmoji(emoji) {
    if (socket && socket.connected) {
        // Emit the emoji to the server
        socket.emit('send-emoji', { emoji });
    } else {
        displaySystemMessage('Not connected to server');
    }
}

// Display a regular message in the chat
function displayMessage(message) {
    const messageDiv = document.createElement('div');
    messageDiv.className = 'message';

    // Format timestamp
    const time = new Date(message.timestamp).toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit'
    });

    messageDiv.innerHTML = `
        <div class="message-header">
            <span class="message-username">${escapeHtml(message.username)}</span>
            <span class="message-time">${time}</span>
        </div>
        <div class="message-emoji">${message.emoji}</div>
    `;

    chatMessages.appendChild(messageDiv);

    // Auto-scroll to the latest message
    scrollToBottom();
}

// Display a system message (user joined/left)
function displaySystemMessage(text) {
    const messageDiv = document.createElement('div');
    messageDiv.className = 'system-message';
    messageDiv.textContent = text;
    chatMessages.appendChild(messageDiv);

    // Auto-scroll to the latest message
    scrollToBottom();
}

// Scroll chat to bottom (latest message)
function scrollToBottom() {
    chatMessages.scrollTop = chatMessages.scrollHeight;
}

// Escape HTML to prevent XSS attacks
function escapeHtml(text) {
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
}

// Initialize the app when DOM is loaded
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
} else {
    init();
}
